/****************************************************************************
*****************************************************************************
**
** File Name
** ---------
**
** LinkList.c
**
*****************************************************************************
*****************************************************************************
**
** Description
** -----------
**
** This source file contains services for implementing linked lists.
**
** Services are defined for managing both singly and doubly linked lists.
** In both cases, a dummy head node containing a forward and a backward link
** is used to avoid special cases of empty lists and other boundary
** conditions.
**
*****************************************************************************
*****************************************************************************
**
** Source Change Indices
** ---------------------
**
** Porting: <none>0----<major>         Customization: <none>0----<major>
**
*****************************************************************************
*****************************************************************************
*/

#include "linklist.h"         /* linked lists public interfaces                  */



/****************************************************************************
**
** Public Services
**
*****************************************************************************
*/

/*---------------------------------------------------------------------------
** LL_PutDoubleLinkList()
**---------------------------------------------------------------------------
*/

void LL_PutDoubleLinkList( void *pHead, void *pNew )
{
   /*
   ** We're inserting the new node at the tail of the list
   ** between the tail node (predecessor of the head node) and the head node.
   */

   /*
   ** Successor of the new node is the head node.
   ** Predecessor of the new node is the predecessor of the head node.
   */

   ((ll_DoubleLinkEntryType*)pNew)->pNext = pHead;

   ((ll_DoubleLinkEntryType*)pNew)->pPrev = ((ll_DoubleLinkEntryType*)pHead)->pPrev;

   /*
   ** Successor of the tail node is the new node.
   ** Predecessor of the head node is the new node.
   */

   ((ll_HeadNodeType*)pHead)->pPrev->pNext = pNew;

   ((ll_HeadNodeType*)pHead)->pPrev = pNew;

} /* end of LL_PutDoubleLinkList() */



/*---------------------------------------------------------------------------
** LL_PutSingleLinkList()
**---------------------------------------------------------------------------
*/

void LL_PutSingleLinkList( void *pHead, void *pNew )
{
   /*
   ** We're inserting the new node at the tail of the list
   ** between the tail node (predecessor of the head node) and the head node.
   */

   /*
   ** Successor of the new node is the head node.
   */

   ((ll_SingleLinkEntryType*)pNew)->pNext = pHead;

   /*
   ** Successor of the tail node is the new node.
   ** Predecessor of the head node is the new node.
   */

   ((ll_SingleLinkEntryType*)((ll_HeadNodeType*)pHead)->pPrev)->pNext = pNew;

   ((ll_HeadNodeType*)pHead)->pPrev = pNew;

} /* LL_PutSingleLinkList() */



/*---------------------------------------------------------------------------
** LL_RemoveDoubleLinkList()
**---------------------------------------------------------------------------
*/

void LL_RemoveDoubleLinkList( void *pNode )
{
   /*
   ** Set the predecessor of the successor of the node to be removed to the
   ** predecessor of the node to be removed.
   */

   ((ll_DoubleLinkEntryType*)((ll_DoubleLinkEntryType*)pNode)->pNext)->pPrev =
      ((ll_DoubleLinkEntryType*)pNode)->pPrev;

   /*
   ** Set the successor of the predecessor of the node to be removed to the
   ** successor of the node to be removed.
   */

   ((ll_DoubleLinkEntryType*)((ll_DoubleLinkEntryType*)pNode)->pPrev)->pNext =
      ((ll_DoubleLinkEntryType*)pNode)->pNext;

} /* LL_RemoveDoubleLinkList() */



/*---------------------------------------------------------------------------
** LL_TakeSingleLinkList()
**---------------------------------------------------------------------------
*/

void *LL_TakeSingleLinkList( void *pHead )
{
   ll_SingleLinkEntryType *pTemp;


   /*
   ** Grab the first entry in the list.
   */

   pTemp = (ll_SingleLinkEntryType*)(((ll_HeadNodeType*)pHead)->pNext);

   /*
   ** Successor of head node is next node in the list.
   */

   ((ll_HeadNodeType*)pHead)->pNext = (void*)(pTemp->pNext);

   /*
   ** Special case if we've just taken the last node off the list:
   ** adjust predecessor of head to point at head, i.e. indicate empty.
   */

   if( ((ll_HeadNodeType*)pHead)->pNext == pHead )
   {
      ((ll_HeadNodeType*)pHead)->pPrev = pHead;
   }

   /*
   ** Return the entry removed from the list.
   */

   return( (void*)pTemp );

} /* end of LL_TakeSingleLinkList() */

