/////////////////////////////////////////////////////////////////////////////
//
// PLXNet.c
//
// socket utilities
//
// Copyright (c) 2011-2012, ProSoft Technology, All Rights Reserved
//
//
// DATE     : Aug/08/2011
// MODIFIED :
// Aug/08/2011 HYU - Created
//  04/03/2012 HYU - Modified the function prototype of SocketCreateConnection().
//                   It will return a positive number if connect fails with error EALREADY or select timeout happens.
//  08/22/2012 HYU - Added the functions for UDP.
/////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <unistd.h>
#include <netinet/tcp.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <net/if.h> 

#include "ocxbpapi.h"

#include "plxnet.h"
#include "nethelper.h"


int SocketInit( void )
{
	return 0;
}

void SocketAbort(socket_t socket)
{
	struct linger ling;
	ling.l_onoff  = 1;
	ling.l_linger = 0;
	setsockopt(socket, SOL_SOCKET, SO_LINGER, &ling, sizeof(ling));
	close(socket);
}

int SocketSend(socket_t s, const void *buf, int len)
{
    int slen, n;
    
    for (slen= 0; slen< len; slen+= n)
    {
        // Try to transfer as much of the remaining data as possible.
        n = send(s, (unsigned char *) buf + slen, len - slen, 0);
        
        // Check EOF.
        if (n == 0)
            return 0;
        
        // Check for other errors.
        if (n == -1)
        {
            // Check for possible blocking.
            if (errno == EAGAIN || errno == ENOBUFS)
            {
                // Wait for the blocking to subside.
                // Did select() succeed?
                // Blocking subsided.  Continue data transfer.
                n = 0;
                continue;
            }
            
            // Other data transfer or select() failures.
            return 0;
        }
    }
    
    return slen;
}

socket_t SocketCreateListener(int port)
{
	int opt_set = 1;
	socket_t listener_socket;
	struct sockaddr_in listener_addr;

	if((listener_socket = SocketOpen(AF_INET,SOCK_STREAM,0)) == -1)
	{
		return INVALID_SOCKET;
	}

	// Set this option so the program can restart very quickly after a shutdown
	if (setsockopt(listener_socket, SOL_SOCKET, SO_REUSEADDR, &opt_set, sizeof(int)) == -1) 
	{
		perror("setsockopt listener SO_REUSEADDR");
		close(listener_socket);
		return INVALID_SOCKET;
	}
	
	// Set this option so the connection is kept alive
	if (setsockopt(listener_socket, SOL_SOCKET, SO_KEEPALIVE , &opt_set, sizeof(int)) == -1) 
	{
		perror("setsockopt listener KEEPALIVE");
		close(listener_socket);
		return INVALID_SOCKET;
	}

	listener_addr.sin_family = AF_INET;
	listener_addr.sin_port = htons(port);
	listener_addr.sin_addr.s_addr = INADDR_ANY;
	if(SocketBind(listener_socket,(struct sockaddr*)&listener_addr,sizeof(struct sockaddr)) == -1)
	{
		close(listener_socket);
		return INVALID_SOCKET;
	}

	if(SocketListen(listener_socket,SOMAXCONN) == -1)
	{
		close(listener_socket);
		return INVALID_SOCKET;
	}

	return listener_socket;
}


socket_t SocketAcceptConnection(socket_t listener_socket)
{
	fd_set allfd;
	fd_set readfd;
	struct timeval tv;
	struct sockaddr new_addr;
	socklen_t new_len = sizeof(new_addr);
	int sock;
	int opt_set = 1;

	FD_ZERO(&allfd);
	FD_SET(listener_socket, &allfd);
	tv.tv_sec = 0;
	tv.tv_usec = 0;

	readfd = allfd;
	if (select(listener_socket + 1, &readfd, NULL, NULL, &tv) <= 0)
	{
		//printf("select failed\n");
		return INVALID_SOCKET;
	}

	if (!FD_SET(listener_socket, &readfd))
	{
		printf("readfd not set\n");
		return INVALID_SOCKET;
	}

	if((sock = SocketAccept(listener_socket,&new_addr,&new_len)) == -1)
	{
		return INVALID_SOCKET;
	}

    // Set to non blocking
    if (fcntl(sock, F_SETFL, O_NONBLOCK) == -1)
    {
        perror("Failed to call fcntl in SocketAcceptConnection");
        close(sock);
        return INVALID_SOCKET;
    }

	if(setsockopt(sock,SOL_SOCKET,SO_KEEPALIVE,(char*)&opt_set,sizeof(opt_set)) == -1)
	{
		close(sock);
		return INVALID_SOCKET;
	}

	if (setsockopt(sock,IPPROTO_TCP,TCP_NODELAY,(char*)&opt_set,sizeof(opt_set)) == -1)
	{
		close(sock);
		return INVALID_SOCKET;
	}

	return sock;
}

socket_t SocketCreateTcp2( const char * ifname )
{
    int opt_set = 1;
    socket_t sock = INVALID_SOCKET;
    int sock_type = SOCK_STREAM;
    
    sock = SocketOpen(AF_INET, sock_type, 0);
    if (sock == -1)
    {
        return INVALID_SOCKET;
    }
    
    // Set to non blocking
    if (fcntl(sock, F_SETFL, O_NONBLOCK) == -1)
    {
        perror("Failed to call fcntl in SocketCreateTcp");
        close(sock);
        return INVALID_SOCKET;
    }

    if (ifname != NULL)
    {
        if( setsockopt( sock, SOL_SOCKET, SO_BINDTODEVICE, ifname, strlen(ifname) ) == -1 )
        {
            perror("setsockopt SO_BINDTODEVICE");
        }
    }
    
    if(setsockopt(sock,SOL_SOCKET,SO_KEEPALIVE,(char*)&opt_set,sizeof(opt_set)) == -1)
    {
        close(sock);
        return INVALID_SOCKET;
    }
    
    if (setsockopt(sock,IPPROTO_TCP,TCP_NODELAY,(char*)&opt_set,sizeof(opt_set)) == -1)
    {
        close(sock);
        return INVALID_SOCKET;
    }
    
    return sock;
}

socket_t SocketCreateTcp()
{
    return SocketCreateTcp2(NULL);
}

static int SocketIsConnected(socket_t sock, fd_set *rd, fd_set *wr, fd_set *ex)
{
    int err;
    int len;

    errno = 0;
    if (!FD_ISSET(sock, rd) && !FD_ISSET(sock, wr))
    {
        return 0;
    }

    if (getsockopt(sock, SOL_SOCKET, SO_ERROR, &err, (socklen_t *)&len) < 0)
    {
        return 0;
    }

    errno = err;

    return err == 0;
}

int SocketCreateConnection(socket_t sock, unsigned long ip, unsigned short port)
{
	fd_set readfd;
    fd_set writefd;
    fd_set execfd;
	struct timeval tv;
	struct sockaddr_in servaddr;
	int rc;

	if (sock == INVALID_SOCKET)
	{
		return -1;
	}

    memset(&servaddr, 0, sizeof(servaddr));
    servaddr.sin_family      = AF_INET;
    servaddr.sin_addr.s_addr = ip;
    servaddr.sin_port        = htons(port);

    rc = connect(sock, (struct sockaddr *)&servaddr, sizeof(servaddr));
    if ( rc == -1 && errno != EINPROGRESS)
    {
        if (errno == EALREADY)
        {
            return 2;
        }

        //perror("Failed to call connect in SocketCreateConnection");
        return -1;
    }
    else if (rc == 0)
    {
    	// already connected
    	return 0;
    }

	FD_ZERO(&readfd);
	FD_SET(sock, &readfd);
    writefd = readfd;
    execfd = readfd;

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	rc = select(sock + 1, &readfd, &writefd, &execfd, &tv);
	if (rc < 0)
	{
		perror("Failed to call select in SocketCreateConnection");
		return -1;
	}
	else if (rc == 0)
	{
		// Time out
		return 1;
	}
	else if (!SocketIsConnected(sock, &readfd, &writefd, &execfd))
    {
#ifdef DEBUG
        perror("Cannot connect to a server");
#endif
        return -1;
    }

	return 0;
}

void se_wait_set(SOCKET_EVENT_MGR *netMgr, socket_t s)
{
	FD_SET(s,&netMgr->m_irfds);
	FD_SET(s,&netMgr->m_iwfds);
	FD_SET(s,&netMgr->m_iefds);

	int i;
    for (i = 0; i < FD_SETSIZE; i++)
    {
        if (netMgr->m_socketArray[i] == -1)
        {
        	netMgr->m_socketCount++;
            netMgr->m_socketArray[i] = s;
            break;
        }
    }

    if (i == FD_SETSIZE)
    {
    	// no space to add
    	return;
    }

    if (netMgr->m_lastSocketIndex < i)
    {
    	netMgr->m_lastSocketIndex = i;
    }

	if (netMgr->m_LastSocket < s)
	{
		netMgr->m_LastSocket = s;
	}
}

void se_wait_clear(SOCKET_EVENT_MGR *netMgr, socket_t s)
{
	int i;
	socket_t sock;

	FD_CLR(s,&netMgr->m_irfds);
	FD_CLR(s,&netMgr->m_iwfds);
	FD_CLR(s,&netMgr->m_iefds);

    for (i = 0; i <= netMgr->m_lastSocketIndex; i++)
    {
        if (netMgr->m_socketArray[i] == s)
        {
        	netMgr->m_socketCount--;
            netMgr->m_socketArray[i] = -1;
            break;
        }
    }

	if (s == netMgr->m_LastSocket)
	{
		netMgr->m_LastSocket = -1;
		for(i=0; i <= netMgr->m_lastSocketIndex; i++)
		{
			sock = netMgr->m_socketArray[i];
			if (netMgr->m_LastSocket < sock)
			{
				netMgr->m_LastSocket = sock;
			}
		}
	}
}

int se_wait_read(SOCKET_EVENT_MGR *netMgr, socket_t s)
{
	return FD_ISSET(s,&netMgr->m_srfds);
}

int se_wait_write(SOCKET_EVENT_MGR *netMgr, socket_t s)
{
	return FD_ISSET(s,&netMgr->m_swfds);
}

int se_wait(SOCKET_EVENT_MGR *netMgr, DWORD dwTimeout)
{
	struct timeval tv;
	tv.tv_sec = (dwTimeout / 1000);
	tv.tv_usec = (dwTimeout % 1000) * 1000;

	// Copy our interesting fds into the set used in select.
	netMgr->m_srfds = netMgr->m_irfds;
	netMgr->m_swfds = netMgr->m_iwfds;
	netMgr->m_sefds = netMgr->m_iefds;
	return select(netMgr->m_LastSocket+1,&netMgr->m_srfds,&netMgr->m_swfds,&netMgr->m_sefds,&tv);
}


int SocketCreateUdp(unsigned short port, const char *ifname, socket_t *sock_p)
{
    socket_t sock;
    struct sockaddr_in my_addr;   // my address information
    int opt;
    
    if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
    {
        perror("SocketCreateUdp");
        return -1;
    }

    if (ifname != NULL)
    {
        if( setsockopt( sock, SOL_SOCKET, SO_BINDTODEVICE, ifname, strlen(ifname) ) == -1 )
        {
            close(sock);
            //perror("setsockopt SO_BINDTODEVICE");
            return -1;
        }
    }
    
    opt = 0;	//disable loopback message to socket
    setsockopt(sock, IPPROTO_IP, IP_MULTICAST_LOOP, &opt, sizeof(opt));

    opt = 1;
    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(int)) == -1) 
    {
        perror("setsockopt SO_REUSEADDR");
        close(sock);
        return -1;
    }

/*
    // Set to non blocking
    if (fcntl(sock, F_SETFL, O_NONBLOCK) == -1)
    {
        perror("fcntl in SocketCreateUdp()");
        return -1;
    }
*/

    opt = 8192;
    
    if ( setsockopt( sock, SOL_SOCKET, SO_SNDBUF, ( char * ) &opt, sizeof( opt ) ) == -1 )
    {
        perror("setsockopt SO_SNDBUF");
    }
    
    opt = 8192;
    
    if ( setsockopt( sock, SOL_SOCKET, SO_RCVBUF, ( char * ) &opt, sizeof( opt ) ) == -1 )
    {
        perror("setsockopt SO_RCVBUF");
    }

    // Set the address to bind to
    my_addr.sin_addr.s_addr = INADDR_ANY; // automatically fill with my IP
    my_addr.sin_family = AF_INET; // host byte order
    my_addr.sin_port = htons(port); // short, network byte order
    // This would be the standard for a service that will run in every port
    memset(&(my_addr.sin_zero), '\0', 8); // zero the rest of the struct
    
    // Proceed to BIND
    if (bind(sock, (struct sockaddr *)&my_addr, sizeof(struct sockaddr)) == -1)
    {
        close(sock);
        perror("SocketCreateUdp");
        return -1;
    }

    *sock_p = sock;

    return 0;
}

int SocketSendUdp(socket_t sock, int flag, unsigned long ip, unsigned short port, const void *buf, int len)
{
    struct sockaddr_in servaddr;
    int opt;

    if (sock == INVALID_SOCKET)
    {
        return -1;
    }

    if(flag & PLX_UDP_BROADCAST)
    {
        ip = INADDR_BROADCAST;

        opt = 1;	//enable broadcasting in case this is a broadcast message
        setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt));
    }
    else
    {
        opt = 0;	//disable broadcasting in case this is a broadcast message
        setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt));
    }
    
    if(flag & PLX_UDP_MULTICAST)
    {
        if ( !IN_MULTICAST( ntohl( ip ) ) )
        {
            return -1;
        }
    }

    memset(&servaddr, 0, sizeof(servaddr));
    servaddr.sin_family      = AF_INET;
    servaddr.sin_addr.s_addr = ip;
    servaddr.sin_port        = htons(port);
    return sendto(sock, buf, len, 0, (struct sockaddr*)&servaddr, sizeof(servaddr));
}

int SocketReadUdp(socket_t sock, unsigned long *ip, unsigned short *port, char *buffer, int max_len)
{
    struct sockaddr_in clientaddr;
    socklen_t len_addr;
    int len;
    
    if(sock == INVALID_SOCKET)
    {
        return -1;
    }

    len_addr = sizeof(clientaddr);
    len = recvfrom(sock, buffer, max_len, MSG_DONTWAIT | MSG_PEEK, (struct sockaddr*)&clientaddr, &len_addr);
    if(len > 0)	//data ready for read
    {
        len_addr = sizeof(clientaddr);
        len = recvfrom(sock, buffer, max_len, 0, (struct sockaddr*)&clientaddr, &len_addr);
        if(ip)
        {
            *ip = clientaddr.sin_addr.s_addr;
        }
        if(port)
        {
            *port = ntohs(clientaddr.sin_port);
        }
    }
    else	//no data ready for read
    {
        len = 0;
    }
    
    return len;
}

int SocketJoinMulticast( socket_t sock, unsigned long addr )
{
    struct ip_mreq  sMreq;

    if ( IN_MULTICAST( ntohl( addr ) ) )
    {
        /* 
        ** No pre-existing connections for this address, need to add a new one 
        */
        
        sMreq.imr_interface.s_addr = htonl( INADDR_ANY );
        
        /* 
        ** fill in the argument structure to join the multicast group 
        ** initialize the multicast address to join 
        */ 
        
        sMreq.imr_multiaddr.s_addr = addr;
        
        /*
        ** First check to see if this socket is initialized. We may have exceeded the
        ** IP_MAX_MEMBERSHIPS limit and not had the option of opening another 
        ** socket.
        */
        if ( setsockopt( sock, IPPROTO_IP, IP_ADD_MEMBERSHIP,
            ( char * ) &sMreq, sizeof( sMreq ) ) == -1 )
        {
            return -1;
        }  /* end if setsockopt  */
        
        /*
        ** keep track of the multicast groups, and the count of clients listening.
        */
        
        return 0;
    }  /* end if ( IN_MULTICAST( tmp_addr.sin_addr.s_addr ) ) */
    
    return -1;
}

int SocketDropMulticast( socket_t sock, unsigned long addr )
{
    struct ip_mreq  sMreq;

    if ( IN_MULTICAST( ntohl( addr ) ) )
    {
        /*
        ** Check to make sure we really joined this multicast group.  If we
        ** didn't this is a definate programming error
        */
        
        sMreq.imr_multiaddr.s_addr = addr;
        /*
        ** Wind River requires the interface address in order to drop
        ** membership in a multicast connection.  INADDR_ANY will return an
        ** error
        */
        sMreq.imr_interface.s_addr = htonl( INADDR_ANY );
        if ( setsockopt( sock, IPPROTO_IP, IP_DROP_MEMBERSHIP,
            ( char * ) &sMreq, sizeof( sMreq ) ) == -1 )
        {
            return -1;
        }  /* end if setsockopt  */
        
        return 0;
    }  /* end if ( IN_MULTICAST( * sTmp_addr.sin_addr.s_addr ) ) */
    
    /*jjw do we really want to return OK if specified address is not Multicast ??*/
    return 0;
}

