/////////////////////////////////////////////////////////////////////////////
// tw-ldm-sample-app.C
//
// This file contains code of ThingWorx-LDM interface sample application for the MVI56E-LDM module.
//
//  This sample code maps ThingWorx things' properties to PLC tags and synchromizes them.
//
// (c) 2019, ProSoft Technology , All Rights Reserved
//
//
/////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>

#include "types-and-variables.h"
#include <sys/select.h>
#include <fcntl.h>

volatile uint16_t scan_counter = 0;

void stop_tw_and_exit(void)
{
	int result;

	close_backplane();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Backplane is closed");	

	tw_ldm_log(TW_LDM_LOG_INFO, "Disconnecting from ThingWorx Server.");
	result = tw_ldm_disconnect();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Disconnected from ThingWorx Server, result = %d", result);
	
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Stopping ThingWorx subsystem.");
	result = tw_ldm_clean();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Stopped ThingWorx subsystem, result = %d", result);

	tw_ldm_free_config(&gl_app_config);

	// set_keyboard(FALSE);

	exit(exit_code);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	sigquit_handler( int sig )
//
//				Input:
//   				sig - is the signal that caused the invocation.
//
//				Description:
// 					This is the normal exit routine for the application.  It is called when the
//					program exits and when the following signals are received: SIGSEGV, SIGQUIT,
//					SIGINT or SIGTERM.  This function will cause the module to exit(not reboot)
//					after the application closes all the resources.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void sigquit_handler( int sig )
{
  const char* reason;
  switch (sig)
  {
  case SIGTERM:
    exit_code = 0;
    reason = "Termination";
    break;
  case SIGKILL:
    exit_code = 0;
    reason = "Kill , unblockable";
    break; 
  case SIGQUIT:
    exit_code = 0;
    reason = "Quit";
    break;
  case SIGINT:
    exit_code = 0;
    reason = "Interrupt";
    break;
  default:
    exit_code = -2;
    reason  = "unknown";
    break;
  }
	
	tw_ldm_log(TW_LDM_LOG_NOTICE, "\nReceived signal # %d [%s], scan counter = %d", sig, reason, scan_counter );

	// Call function stop_tw_and_exit only once, to prevent re-entrances due to exceptions during stopping.
	if (!Done)
	{
		Done = TRUE;
		stop_tw_and_exit();
	}
	else
		exit(-3);
}

// Buffer to convert OCX error result to string.
char error_string[80];

int get_current_year()
{
	time_t t = time(NULL);
	struct tm tm = *localtime(&t);
	return tm.tm_year;
}

int sync_system_time_with_plc()
{
	// Setting to Null is special case: when set to NULL, system time will be set to the time returned from the PLC.
	OCXCIPWCT* pWCT = NULL;
	OCXCIPWCT  current_plc_time;
	memset(&current_plc_time, 0, sizeof(current_plc_time));

	const WORD timeout_in_ms = 100;

	// First, check if current PLC time looks correct. Criteria is: year must be 2019 or later.
	OCXAPIENTRY get_time_result = OCXcip_GetWCTime(OCX_Handle, (BYTE*)gl_app_config.plc_path, &current_plc_time, timeout_in_ms);
	if (get_time_result == OCX_SUCCESS || get_time_result == OCX_ERR_REMOTE_STS_OFFSET)
	{
		if (current_plc_time.SystemTime.wYear >= 2019)
		{
			// PLC time seems set correctly, can be used to sync system time.
			OCXAPIENTRY set_time_result = OCXcip_GetWCTime(OCX_Handle, (BYTE*)gl_app_config.plc_path, pWCT, timeout_in_ms);
			if (set_time_result == OCX_SUCCESS || set_time_result == OCX_ERR_REMOTE_STS_OFFSET)
			{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "GetWCTime succeeded, system time should be synchronized with PLC time now.");
				return 0;
			}
			else
			{
				OCXcip_ErrorString(set_time_result, error_string);
				tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_GetWCTimeUTC failed, result = [%d:%s]. If system time is not correct, this might prevent connection to the ThingWorx server.", set_time_result, error_string);
				return (int)set_time_result;
			}
		}
		else
		{
			tw_ldm_log(TW_LDM_LOG_NOTICE, "PLC time does not look correct (year is %d), will not use it to set system time.", current_plc_time.SystemTime.wYear);
			return -1;
		}
	}
	else
	{
		OCXcip_ErrorString(get_time_result, error_string);
		tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_GetWCTime failed, result = [%d:%s]. If system time is not correct, this might prevent connection to the ThingWorx server.", get_time_result, error_string);
		return (int)get_time_result;
	}
}

uint32_t backplane_connect_start_time;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// 		main program()
//			Input:
//				argc - argument count.
//				argv - list of arguments.
//
//			Description:  This sample establishes a connection with the PLC, reads the tag database information, starts ThingWorx threads, reads PLC data and sets ThingWorx property values.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

int main( int argc, char * argv[] )
{
  int						      result;
  OCXCIPVERSIONINFO		version_info;
  int						      last_connected = FALSE;
  char*               config_file_name = "config.json";
  static int          printed_connection_error = 0;
	uint32_t						last_tick = 0;
  uint32_t            last_status_print = 0;
	char pre_is_connected = -1;
	char pre_is_connecting = -1;

  printf("ThingWorx-MVI56e-LDM Interface Sample Application version 1.0\n");

  // set up special event handlers.
  signal( SIGSEGV, sigquit_handler );
  signal( SIGQUIT, sigquit_handler );
  signal( SIGINT,  sigquit_handler );
  signal( SIGTERM, sigquit_handler );

	if ( argc > 1 )
	{
  	if (argv[1][0] == '-' && (argv[1][1] == 'h' || argv[1][1] == 'H'))
  	{
    	printf("\n Usage:\n tw-ldm-sample-app-mvi56 [config-file-name]\n  Communicates with ThingWorx Server and ControlLogix processor,");
    	printf("\n maps thing properties to PLC tags and synchronizes their values.");
    	printf("\n Optional parameter config-file-name: name of configuration file, by default \"config.json\"\n\n");

    	exit(-1);
  	}
  	else
    	config_file_name = argv[1];
	}

  // Load configuration file.
  printf("Loading configuration settings from file [%s]\n", config_file_name);
  result = tw_ldm_initialize(config_file_name, NULL);
  if (result != 0)
  {
    tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to initialize ThingWorx subsystem, exiting");
    exit(result);
  }
	
	// Set keyboard mode to non-blocking:
	fcntl(0, F_SETFL, O_NONBLOCK);

	while (Done == FALSE)
	{
		if (!Backplane_Connected && !Backplane_Connecting)
		{
			Backplane_Connecting = TRUE;
			backplane_connect_start_time = get_tick_count();
			//  open access to backplane and hardware
			result = open_backplane();
			if (result == 0)
			{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "open_backplane succeeded");
			}
			else
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane failed with result = %d\n", result);
			}

			if (result == 0)
			{
				// initialize LED
				OCXcip_SetModuleStatus(OCX_Handle, OCX_MODULE_STATUS_OK);
				OCXcip_SetUserLED(OCX_Handle, OCX_LED_STATE_OFF);
				OCXcip_SetLED3(OCX_Handle, OCX_LED_STATE_OFF);

				// read Version information of the Backplane firmware components.
				result = OCXcip_GetVersionInfo(OCX_Handle, &version_info);
				if (result == OCX_SUCCESS)
				{
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info API Series:             %d", version_info.APISeries);
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info API Revision:           %d", version_info.APIRevision);
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info BP Engine Series:       %d", version_info.BPEngSeries);
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info BP Engine Revision:     %d", version_info.BPEngRevision);
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info BP Dev Driver Series:   %d", version_info.BPDDSeries);
					tw_ldm_log(TW_LDM_LOG_DEBUG, "Version Info BP Dev Driver Revision: %d", version_info.BPDDRevision);
				}
				else
				{
					OCXcip_ErrorString(result, error_string);
					tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_GetVersionInfo failed, result = [%d:%s]", result, error_string);
					exit(result);
				}

				Display("RUN ");
			}
		}
		else
		{
			// If for some reason backplane stays not connected for long time, and is not connecting, 
			// then reset "connecting" flag, so new connection can be started:
			if ((get_tick_count() - backplane_connect_start_time) > 1000 * 30)
			{
				Backplane_Connecting = FALSE;
			}
		}

		uint32_t current_tick = get_tick_count();
		char is_connected;
		char is_connecting;

		int key = 0;  
		char c = 0;

		scan_counter++;

		if (!Backplane_Connected)
		{
			current_status = "Not connected";
			if (!printed_connection_error)
			{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "Waiting for backplane connection");
				printed_connection_error = 1;
			}
			// sleep for 250 milliseconds, sharing the CPU while waiting for a connection
			usleep(250 * 1000);
		}
		else
		{
			// connection is established.
			if (last_connected != TRUE)
			{
				last_connected = TRUE;

				OCXcip_SetModuleStatusWord(OCX_Handle, OCX_MOD_STATUS_CONNECTED, OCX_MOD_STATUS_STATE_MASK);
				OCXcip_SetModuleStatusWord(OCX_Handle, OCX_MOD_STATUS_OWNED, OCX_MOD_STATUS_OWNED_MASK);

				// open and build database of controller tags
				result = open_tag_dbase(gl_app_config.plc_path);
				if (result != OCX_SUCCESS)
				{
					exit_code = -1;
					last_connected = FALSE;
					usleep(10000);
					continue;
				}

				switch (gl_app_config.sync_time_with_plc)
				{
				case 0: 
					// Sync if system time looks not set correctly:
					if (get_current_year() < 2019)
						sync_system_time_with_plc();
					break;
				case 1:
				{
					// Sync once.
					static int number_of_syncs = 0;
					if (number_of_syncs == 0)
					{
						if (sync_system_time_with_plc() == 0)
						{
							number_of_syncs++;
						}
					}
					break;
				}
				case 2:
					// Sync always:
					sync_system_time_with_plc();
					break;
				}
			}

			// check and log the controller status, if it changed.
			check_controller_status(TRUE, FALSE);

			if (key == 's' || key == 'S')
			{
				print_rack_information();
				print_database_symbols();
			}

			usleep(1000);
			printed_connection_error = 0;
		}

		if ((current_tick - last_tick) > 1000)
		{
			last_tick = current_tick;
			
			key = read(0, &c, 1);
			key = c;

			is_connected = tw_ldm_is_connected();
			is_connecting = tw_ldm_is_connecting();
			if ((pre_is_connected != is_connected) || (pre_is_connecting != is_connecting) || ( (current_tick - last_status_print) > (gl_app_config.status_print_interval * 1000)) )
			{
  			last_status_print = current_tick;
				tw_ldm_log(TW_LDM_LOG_NOTICE, "ThingWorx Server status: %s%s. Scan counter = %d", (is_connected) ? "connected" : " disconnected", (is_connecting ? ", connection in progress" : ""), scan_counter);
				pre_is_connected = is_connected;
				pre_is_connecting = is_connecting;
			}

  		if (!is_connected && !is_connecting)
  		{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "Trying to connect to ThingWorx Server");
    		tw_ldm_disconnect();
				
        // TODO: potentially it would be possible to re-load configuration, but it is not thread-safe, background polling function might be accessing it from ThignWorx thread.
        // So not re-loading configuration for now.
				//tw_ldm_free_config(&gl_app_config);
				//load_and_parse_configuration_file(config_file_name, &gl_app_config);

				usleep(1000000);
    		int connect_result = tw_ldm_connect();
				if (connect_result != 0)
				{
					if (get_current_year() < 2019)
						sync_system_time_with_plc();
				}
  		}
		}

		if ( (key == 27) || (key == 'x') || (key == 'X') || (key == 'c') || (key == 'C') )
		{
			exit_code = 0;
			tw_ldm_log(TW_LDM_LOG_NOTICE, "Escape key pressed, exiting");
			Done = TRUE;
			break;
		}
	}

	stop_tw_and_exit();
	return -1;
}
