"""
"""
import json, requests
import os
from os import path
from datetime import datetime
from time import time

JSON_HEADERS = {"Content-type": "application/json", "Accept": "*/*"}
TREND_DATA_DURATIONS = [
        "Last 5 min",
        "Last 30 min",
        "Last hour",
        "Last 6 hours",
        "Last 12 hours",
        "Last day",
        "Last 5 days",
        "Last week",
        "Last month",
        "Last year",
        "Custom Dates"
    ]
        
class Plx5xConnection:
    def __init__(self, url, print_en=True, log_requests=False, log_responses=False, log_dir=None):
        """
        Python interface for PLX51 data logger REST API.  Provides methods for each available API call and a few
        extra usability bonuses.
        
        :param str url: IP address of PLX51 module.  Must be reachable with current network settings
        :param bool print_en: Flag for enabling/disabling printing of JSON requests/replies to stdout
        :param bool log_requests: Enable logging of JSON requests
        :param bool log_responses: Enable logging of JSON responses
        :param str log_dir: Path in which to store log files. If None, defaults to cwd (dir from which script is run)
        
        Some considerations if adapting this script for more advanced use:
        - WARNING: If the module is not running (getGenSts returns Running=0) many of these commands will not work or
            will have confusing behavior.
        
        - WARNING: If adapting for use with large amounts of data or long-running collection of data,
            the logging system implemented here will quickly fall flat on its face.  Python's built-in
            logging library provides rotating file handles that can help to address these issues
        """
        # Perform some very very basic checks on input URL
        if url.startswith("http://"):
            self.url = url
        else:
            self.url = "http://{}".format(url)
        
        # Collet output/log settings and set up logging
        self._print_en = print_en
        self._log_requests = log_requests
        self._log_responses = log_responses
        self._log_dir = log_dir or os.getcwd()
        self._setup_json_logging()

    @property
    def print_en(self):
        return self._print_en
        
    @print_en.setter
    def print_en(self, val):
        self._print_en = bool(val)

    @property
    def log_requests(self):
        return self._log_requests
        
    @log_requests.setter
    def log_requests(self, val):
        if self._log_dir is not None:
            self._log_requests = bool(val)
        else:
            raise Exception("Log flag is enabled, but log directory is not set! Set log_dir before enabling logging")
        
    @property
    def log_responses(self):
        return self._log_responses
        
    @log_responses.setter
    def log_responses(self, val):
        if self._log_dir is not None:
            self._log_responses = bool(val)
        else:
            raise Exception("Log flag is enabled, but log directory is not set! Set log_dir before enabling logging")
        
    @property
    def log_dir(self):
        return self._log_dir
        
    @log_dir.setter
    def log_dir(self, val):
        self._log_dir = val
        self._setup_json_logging()


    def _verify_rest_response(self, resp):
        """
        Accepts requests Response object from GET/POST and verifies HTTP was OK and 
        that the requested call was valid.
        """
        # Check that we got HTTP 200/OK
        if resp.status_code != 200:
            raise Exception("Invalid status code from device {}. Got {}".format(self.url, resp.status_code))
        
        # Check that JSON response indicates a valid command was sent
        elif resp.json()["header"]["messageType"] == "resInvalidRequest":
            raise Exception("Device at {} got invalid command. See log for details".format(self.url))

    def _setup_json_logging(self):
        """
        Create and check session log files
        """
        if not os.path.exists(self.log_dir):
            os.mkdir(self.log_dir)      # Note: This will not create recursive dirs

        seconds_ts = int(time())
        session_file = "Session{}.txt".format(seconds_ts)
        self.session_log_path = os.path.join(self.log_dir, session_file)

        # Touch the session file and raise access errors.  Note we're not checking
        try:
            open(self.session_log_path,'w').close()
        except Exception as e:
            raise Exception("Could not create session log at {}".format(self.session_log_path), e)

    def perform(self, request_body):
        """
        Wrappers around generic REST API access for the PLX51.
        :param request_body: Dict containing request data that will be converted to JSON and sent.
        :return: Dict of REST API response message content.
        """
        # Log request as necessary
        self.output_json(request_body);
        
        # Perform POST to call REST API in a single-use TCP session
        resp = requests.post(
            self.url,
            data=json.dumps(request_body),
            headers=JSON_HEADERS
        )
        # Log reply as necessary
        self.output_json(resp.json())
        
        # Verify that command was processed correctly and reply was recieved
        self._verify_rest_response(resp)

        return resp.json()

    def get_general_status(self):
        """
        Request general status
        :return: General status as in PLX51 User Manual
        """
        request_body = {
            "header": {
                "messageType": "reqGenSts"
            },
            "requestData": {}
        }

        response = self.perform(request_body)
        return response["responseData"]

    def get_cache_statistics(self):
        """
        Request cache statistics
        :return: Cache stats as in PLX51 User Manual
        """
        request_body = {
            "header": {
                "messageType": "reqCacheStats"
            },
            "requestData": {}
        }

        response = self.perform(request_body)
        return response["responseData"]

    def unload_log_index_update(self, log_index):
        """
        Sets the UnloadIdx flag in the PLX51
        Log index value can be checked with `get_general_status`
        :return: None
        """
        request_body = {
            "header": {
                "messageType": "reqUnloadIdxUpdate"
            },
            "unloadLogIndex": log_index
        }

        self.perform(request_body)

    def log_index_reset(self):
        """
        Reset the log index
        Log index value can be checked with `get_general_status`
        :return: None
        """
        request_body = {
            "header": {
                "messageType": "reqLogIndexReset"
            },
            "requestData": {}
        }

        self.perform(request_body)

    def get_cache_records(self, log_index, record_count):
        request_body = {
            "header": {
                "messageType": "reqCacheRecords"
            },
            "requestData": {
                "logIndex": log_index,
                "recordCount": record_count
            }
        }

        response = self.perform(request_body)
        return response["responseData"]


    def get_tag_names(self, tag_idx, tag_count):
        request_body = {
            "header": {
                "messageType": "reqGetTagNames"
            },
            "requestData": {
                "tagNameIndex": tag_idx,
                "tagNameCount": tag_count
            }
        }
        if tag_count > 4:
            raise ValueError("Attempted to read more than 4 tag names in one request. Tag_count must be 1-4")
        response = self.perform(request_body)
        return response["responseData"]

    def get_trend_data_utc(self, duration_str=None, tag_idxs=[], start_time=None, stop_time=None):
        return self.get_trend_data(duration_str, tag_idxs, start_time, stop_time, utc=True)

    def get_trend_data(self, duration_str=None, tag_idxs=[], start_time=None, stop_time=None, utc=False):
        request_body = {
            "header":{
                "messageType": None     # reqTrendData or reqTrendDataUTC
            },
            "requestData":{
                "command": "Start",
                "duration": None,
                # start_time and stop_time only included if duration is set to Custom Dates
                "extractedTags": None
                # tag Idx N tags only present when in "Tags 1 to 5" mode
            }
        }

        # Determine which API command to use based on UTC flag
        if utc:
            request_body["header"]["messageType"] = "reqTrendDataUTC"
        else:
            request_body["header"]["messageType"] = "reqTrendData"

        # Determine tag extracted tag value and populate tag idx values if in "Tags 1 to 5" mode
        if len(tag_idxs) == 0:
            request_body["requestData"]["extractedTags"] = "All tags"
        elif len(tag_idxs) > 5:
            raise ValueError("GetTrendData: Invalid number of tags requested. Request 1-5")
        else:
            request_body["requestData"]["extractedTags"] = "Tags 1 to 5"
            # Pad the tag idx list to 5 elements as required by data logger API with -1 (no tag value)
            tag_idxs_padded = tag_idxs + [-1] * (5 - len(tag_idxs))
            for idx, t_val in enumerate(tag_idxs_padded):
                request_body['requestData'].update({"tag Idx {}".format(idx+1): t_val})

        # Determine time settings
        if duration_str is None or duration_str == "Custom Dates":
            request_body["requestData"]["duration"] = "Custom Dates"

            if start_time is None or stop_time is None:
                raise ValueError("If no duration string is provided, start_time and stop_time must be specified")
            
            # Use start/stop times for Custom Dates requests.  Can be improved by checking both std and utc times for validity
            # Normal TrendData and TrendDataUTC requests have different stop/start keys
            if utc:
                request_body["requestData"]["startUTC"] = int(start_time)
                request_body["requestData"]["stopUTC"] = int(stop_time)
            else:
                request_body["requestData"]["startTime"] = start_time
                request_body["requestData"]["stopTime"] = stop_time
                
        elif duration_str in TREND_DATA_DURATIONS:
            # Insert duration string into JSON request
            request_body["requestData"]["duration"] = duration_str
        else:
            raise ValueError("'{}' is not a valid duration specification".format(duration_str))

        ### Parsing done.  Now onto sending/getting json data in a session
        # Init results variable and output request text as configured
        trend_data = []
        self.output_json(request_body)
        
        # Send request for trend data and poll until trend data is received
        with requests.Session() as session:
            # Update headers for the upcoming operations
            session.headers.update(JSON_HEADERS)
           
            # Start getting trend data (Start command)
            init = session.post(self.url, data=json.dumps(request_body))
            init_json = init.json()
            
            # Output data to stdout/logs and save in result array
            self.output_json(init_json)
            trend_data.append(init_json)

            self._verify_rest_response(init)
            resp_status = init_json["responseData"]["status"]

            # Change json request to "Poll" command to continue getting trend data until
            # "Last Packet" is found in JSON response from the data logger
            request_body["requestData"]["command"] = "Poll"
            while resp_status != "Last Packet":
                self.output_json(request_body)
                
                poll = session.post(self.url, data=json.dumps(request_body))
                poll_json = poll.json()
                
                # Output data to stdout/logs and save in result array
                self.output_json(poll_json)
                trend_data.append(poll_json)

                # Check response is valid and continue on polling
                self._verify_rest_response(poll)
                resp_status = poll_json['responseData']['status']

        return trend_data

    ### Logging!
    def output_json(self, json_dict):
        """
        Direct JSON to various outputs as necessary per instance settings.
        This logging system would be greatly improved with the inclusion/use of
        python's logging module.  The below is used for demonstration only.
        """
        if self.print_en:
            print(json.dumps(json_dict, indent=4))

        if self.log_requests and "requestData" in json_dict:
            self.log_text("REQUEST: {}\n".format(datetime.now()))
            self.log_json_obj(json_dict)
            
        elif self.log_responses and "responseData" in json_dict:
            self.log_text("RESPONSE: {}\n".format(datetime.now()))
            self.log_json_obj(json_dict)
            
    def log_json_obj(self, data_dict):
        """ Write JSON object (from dict) to the session log file """
        # open the session log file and append json
        with open(self.session_log_path,'a') as f:
            f.write(json.dumps(data_dict, indent=4))
            f.write("\n\n")
            
    def log_text(self, text):
        """ Write target text to the session log file. """
        with open(self.session_log_path,'a') as f:
            f.write(text)

    ### EXTRAS - Features for requesting data by tag name instead of tag index.
    def get_all_tag_names(self):
        """
        Collect and cache all tag names defined in the PLX51 module config.
        This method populates the `self.tags` attribute to link tag names against
        their indexes.
        `Plx5xConnection.get_trend_data_by_tag_name` uses these cached tags to select by name.
        """
        request_body = {
            "header": {
                "messageType": "reqGetTagNames"
            },
            "requestData": {
                "tagNameIndex": None,
                "tagNameCount": None
            }
        }
        
        tags = {}
        config_crc = self.get_general_status()["ConfigCRC"]
        with requests.Session() as session:
            session.headers.update(JSON_HEADERS)

            # Get tags 1 by 1 to simplify example. Blocks of up to 4 tags per requests are supported.
            # This is separated from the main `perform` method so a single TCP session can be used for
            # all requests.
            for tag_block in range(200):        # 200 Max tags supported by PLX51plus232
                request_body["requestData"]["tagNameIndex"] = tag_block
                request_body["requestData"]["tagNameCount"] = 1
                
                resp = session.post(self.url, data=json.dumps(request_body))
                tags_section = resp.json()["responseData"]["tags"]
                
                # Grab the first tag in the results (there should be only one requested above)
                # Once we start getting 0-length results, assume we have reached the end of the tags
                # list and break.
                if len(tags_section) >= 1:
                    tags[tags_section[0]["tagName"]] = {
                        "idx": tags_section[0]["tagId"],
                        "dtype": tags_section[0]["tagDataType"]
                    }
                else:
                    break

        after_crc = self.get_general_status()["ConfigCRC"]
        if config_crc != after_crc:
            raise Exception("Configuration CRC changed while getting tags!  Make sure no config changes are made mid-request.")
            
        return tags

    def get_trend_data_by_tag_name(self, tag_names, duration_str=None, start_time=None, stop_time=None, utc=False):
        all_tags = self.get_all_tag_names()
        idx_list = []
        # Check tag count.  If too few tags are passed, they'll be padded by the main 
        # get_trend_data method.
        if len(tag_names) > 5:
            raise ValueError("Too many tags requested!")
            
        for t in tag_names:
            if t not in all_tags:
                raise ValueError("Cannot read tag {} - not monitored by the module.  Check module config.".format(t))
                
            idx_list.append(self.tags[t]["idx"])
            
        # Call main get_trend_data with tags translated.
        return self.get_trend_data(
            all_tags=False,
            tag_idxs=idx_list,
            duration_str=duration_str,
            start_time=start_time,
            stop_time=stop_time,
            utc=utc
        )

