//***	SERPROT.H -- Remote Disk Serial Protocol Definitions.
//
//1.	Functional Description.
//	This include file contains the definitions needed by the remote
//	disk driver and associated server utility.
//
//2.	Modification History.
//	S. E. Jones	95/03/17.	#6-XL, original.
//
//3.	NOTICE: Copyright (C) 1995 General Software, Inc.
//

#ifndef _SERPROT
#define _SERPROT

#include "..\inc\bpb.h"                 // BIOS parameter block definition.

//
// Protocols & formats.
//

#define FIRM_ACK	0		// 1 enables handshakes, 0 disables.
#define MAX_PINGS	50		// number of ping-pongs.
#define PING_REQ	0x55		// solicit a PING_RSP char from host.
#define PING_RSP	0xaa		// the response to a PING_REQ from target.
#define PING_END	0x66		// we are finished pinging.
#define PING_ENDACK	0x77		// host acknowledgement.

//
// To send a packet of data, we send a MSG_REQ byte followed by a MSGHDR
// written out as individual bytes, followed by any encapsulated data.	We
// expect a MSG_ACK byte from the recipient saying that data were received
// successfully.  If we don't get this byte, then the transport failed, so
// retry the request a predetermined number of times again before failing.
//

#define MSG_REQ         0x41		// request to send a message.
#define MSG_GO		0x42		// reply that send is okay.
#define MSG_ACK         0x43		// received message & data successfully.
#define MSG_NAK         0x44		// received garbled message; resend.
#define MSG_CHOUT	0x45		// fast console out.
#define MSG_CHOUT_ACK	0x46		// acknowledgement to fast console out.

#define MAX_ATTEMPTS	2		// how many times to try to send data.
#define MAX_DATASIZE	1024		// max size of user data is 1K.

typedef struct _MSGHDR {
    USHORT Length;			// msg len, including header & user data.
    UCHAR Checksum;			// checksum over user data.
} MSGHDR, *PMSGHDR;

//
// High-level protocols.
//

#define MSG_DRIVE_DETECT_REQ	0	// drive detect request.
#define MSG_DRIVE_DETECT_RSP	1	// drive detect response (with BPB).

#define MSG_INT16_REQ		2	// keyboard BIOS request.
#define MSG_INT16_RSP		3	// keyboard BIOS response.

#define MSG_INT10_REQ		4	// video BIOS request.
#define MSG_INT10_RSP		5	// video BIOS response.

#define MSG_READ_SECTOR_REQ	6	// read 512-byte sector from drive req.
#define MSG_READ_SECTOR_RSP	7	// read 512-byte sector from drive rsp.

#define MSG_WRITE_SECTOR_REQ	8	// write 512-byte sector from drive req.
#define MSG_WRITE_SECTOR_RSP	9	// write 512-byte sector from drive rsp.

#define MSG_IDENTIFY_HOST_REQ	10	// host ID & capabilities request.
#define MSG_IDENTIFY_HOST_RSP	11	// host ID & capabilities response.

#define MSG_NOTIFY_HOST_REQ	12	// notify host of condition (request).
#define MSG_NOTIFY_HOST_RSP	13	// response (basically, an acknowledgement).

//
// DRIVE DETECT protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_DRIVE_DETECT message	 =>
//				 <= RSP_DRIVE_DETECT message
//

typedef struct _REQ_DRIVE_DETECT {
    USHORT CommandCode;                 // always MSG_DRIVE_DETECT_REQ.
} REQ_DRIVE_DETECT, *PREQ_DRIVE_DETECT;

typedef struct _RSP_DRIVE_DETECT {
    USHORT ResponseCode;		// always MSG_DRIVE_DETECT_RSP.
    BOOLEAN DriveAvailable;		// TRUE if drive service available.
    BPB HostDriveBpb;			// only meaningful if above is TRUE.
} RSP_DRIVE_DETECT, *PRSP_DRIVE_DETECT;

//
// READ SECTOR protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_READ_SECTOR message	 =>
//				 <= RSP_READ_SECTOR message
//

typedef struct _REQ_READ_SECTOR {
    USHORT CommandCode;                 // always MSG_READ_SECTOR_REQ.
    USHORT Unit;
    ULONG SectorNumber;
} REQ_READ_SECTOR, *PREQ_READ_SECTOR;

typedef struct _RSP_READ_SECTOR {
    USHORT ResponseCode;		// always MSG_READ_SECTOR_RSP.
    USHORT Status;			// 0 = success, else DEVERR_ status code.
    UCHAR Data [512];			// data in sector.
} RSP_READ_SECTOR, *PRSP_READ_SECTOR;

//
// WRITE SECTOR protocol:
//
// TARGET			    HOST
// =============================    ==================================
// SYSMSG (MSG_WRITE_SECTOR_REQ) =>
// REQ_WRITE_SECTOR message	 =>
//				 <= SYSMSG (MSG_WRITE_SECTOR_RSP)
//				 <= RSP_WRITE_SECTOR message
//

typedef struct _REQ_WRITE_SECTOR {
    USHORT CommandCode;                 // always MSG_WRITE_SECTOR_REQ.
    USHORT Unit;
    ULONG SectorNumber;
    UCHAR Data [512];			// data to be written.
} REQ_WRITE_SECTOR, *PREQ_WRITE_SECTOR;

typedef struct _RSP_WRITE_SECTOR {
    USHORT ResponseCode;		// always MSG_WRITE_SECTOR_RSP.
    USHORT Status;			// 0 = success, else DEVERR_ status code.
} RSP_WRITE_SECTOR, *PRSP_WRITE_SECTOR;

//
// INT 16h protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_INT16 message		 =>
//				 <= RSP_INT16 message
//

typedef struct _REQ_INT16 {
    USHORT CommandCode;                 // always MSG_INT16_REQ.
    USHORT ax, bx, cx, dx, fl;
} REQ_INT16, *PREQ_INT16;

typedef struct _RSP_INT16 {
    USHORT ResponseCode;		// always MSG_INT16_RSP.
    USHORT ax, bx, cx, dx, fl;
} RSP_INT16, *PRSP_INT16;

//
// INT 10h protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_INT10 message		 =>
//				 <= RSP_INT10 message
//

typedef struct _REQ_INT10 {
    USHORT CommandCode;                 // always MSG_INT10_REQ.
    USHORT ax, bx, cx, dx;
} REQ_INT10, *PREQ_INT10;

typedef struct _RSP_INT10 {
    USHORT ResponseCode;		// always MSG_INT10_RSP.
    USHORT ax, bx, cx, dx;
} RSP_INT10, *PRSP_INT10;

//
// IDENTIFY HOST protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_IDENTIFY_HOST message	 =>
//				 <= RSP_IDENTIFY_HOST message
//

typedef struct _REQ_IDENTIFY_HOST {
    USHORT CommandCode;                 // always MSG_IDENTIFY_HOST_REQ.
    USHORT Flags;			// bitflags, defined below.

#define REQ_IDENTIFY_HOST_FLAGS_SECURITY 0x0001 // security required.
#define REQ_IDENTIFY_HOST_FLAGS_NOACK	 0x0002 // handshaking disabled.

    USHORT SecurityCode;		// compressed password if security set.
} REQ_IDENTIFY_HOST, *PREQ_IDENTIFY_HOST;

typedef struct _RSP_IDENTIFY_HOST {
    USHORT ResponseCode;		// always MSG_IDENTIFY_HOST_RSP.
    USHORT HostId;			// constant for the host session.
    USHORT Flags;			// bitflags, defined below.

#define RSP_IDENTIFY_HOST_FLAGS_SECURITY 0x0001 // security supported.
#define RSP_IDENTIFY_HOST_FLAGS_NOACK	 0x0002 // handshaking disabled.

    USHORT SecurityCode;		// compressed password if security set.
} RSP_IDENTIFY_HOST, *PRSP_IDENTIFY_HOST;

//
// NOTIFY HOST protocol:
//
// TARGET			    HOST
// =============================    ==================================
// REQ_NOTIFY_HOST message	 =>
//				 <= RSP_NOTIFY_HOST message
//

typedef struct _REQ_NOTIFY_HOST {
    USHORT CommandCode;                 // always MSG_NOTIFY_HOST_REQ.
    USHORT Condition;			// condition code (notification code).

#define REQ_NOTIFY_HOST_CONDITION_NONE	  0 // no action (keepalive).
#define REQ_NOTIFY_HOST_CONDITION_SECFAIL 1 // security failure.

} REQ_NOTIFY_HOST, *PREQ_NOTIFY_HOST;

typedef struct _RSP_NOTIFY_HOST {
    USHORT ResponseCode;		// always MSG_NOTIFY_HOST_RSP.
} RSP_NOTIFY_HOST, *PRSP_NOTIFY_HOST;

#endif // _SERPROT
