/*
// PROGRAM NAME:  TIMER.C.
//
// FUNCTIONAL DESCRIPTION.
//	This program is a simple example of how the Embedded DOS timer
//	kernel object works by repeatedly expiring a timer every specified
//	number of milliseconds (a command line parameter).  This is used
//	in conjunction with an oscilloscope to verify the kernel's timer object.
//
//	In order to see this program work, you'll need to hook up a scope to
//	the first LPT port in your system.  If you're not sure which LPT port
//	is going to be used, use DOS DEBUG to dump memory at 40:8 and the first
//	word of memory at that address is the LPT port.  Since this routine
//	toggles all bits in the LPT port, you can choose any bit for I/O.
//
// MODIFICATION HISTORY.
//	S. E. Jones	95/11/14.	#6-XL v40, original.
//
// INPUTS.
//	none.
//
// OUTPUTS.
//	none.
//
// COMMAND LINE ARGUMENTS.
//	none.
//
// NOTICE:  Copyright (C) 1995 General Software, Inc.
*/

#define VOID void
#define UCHAR unsigned char
#define USHORT unsigned short
#define ULONG unsigned long
#define FALSE 0
#define TRUE (!FALSE)

#include <stdio.h>

#include "..\inc\kernel.h"	// kernel API definitions.

USHORT PortNo;
USHORT TimerHandle;
USHORT ExpireTime = 110;	// by default, use 110 milliseconds.
UCHAR Pins=0x00;		// start with pins reset.

VOID _loadds TimerFunction (USHORT Value)
{
    StartTimer (TimerHandle, ExpireTime);

    Pins = ~Pins;
    outp (PortNo, Pins);
} // TimerFunction

VOID main (int argc, char *argv[])
{
    USHORT *PortPtr=(USHORT *)0x00400008L;

    if (argc > 2) {
	printf ("Usage: TIMER <milliseconds>\n");
	exit (1);
    }
    if (argc == 2) {
	ExpireTime = atoi (argv [1]);
    }
    if (ExpireTime == 0) {
	printf ("The expiration time is specified in milliseconds between 1 and 65535.\n");
	exit (2);
    }

    PortNo = *PortPtr;
    if (PortNo == 0) {
	printf ("No parallel ports are available to this program.\n");
	exit (3);
    }

    if (!AllocateTimer (&TimerHandle, 0, TimerFunction)) {
	printf ("Unable to allocate Embedded DOS timer object.\n");
	exit (4);
    }

    printf ("You need an oscilloscope to watch this program function.\n");
    printf ("Toggling pins on LPT1 port attached to I/O location %04x.\n", PortNo);
    printf ("Pins will reverse every %u milliseconds.\n", ExpireTime);
    printf ("Your results may vary depending on the scheduling granularity in OEM.INC.\n");
    printf ("Press any key to stop this test.\n");

    StartTimer (TimerHandle, ExpireTime);

    while (!kbhit ()) {
	PassTimeSlice ();	// call the kernel function.
    }

    StopTimer (TimerHandle);	// in case it fires during deallocation.
    DeallocateTimer (TimerHandle); // remove the timer from the system.

    getch ();			// eat the character from the kbhit above.
    exit (0);			// successfully return to Embedded DOS.
} // timer.c
