/**  $Revision:   1.0  $
/**/
/************************************************************************
*
*   Title: Sample Code for MVI API
*
*   Abstract:
*
*   This sample code illustrates a variety of MVI API functions,
*   while also performing a useful function.  This program implements
*   an interactive diagnostic utility that can be used to verify
*   communications between the MVI module and the controller.
*
*   Environment:    MVI Communications Module
*                   MVI API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*                   Copyright (c) 1998-2000 Online Development, Inc.
*
************************************************************************/

/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <conio.h>
#include    <stdlib.h>
#include    <math.h>
#include    <dos.h>
#include    <ctype.h>
#include    <string.h>
#include    "mvibpapi.h"
#include    "mvispapi.h"

/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void BPErrorExit(int);
void SPErrorExit(void);
void ComPutStr(char *);
void ComGetStr(char *);
void ParseCmdLine(int, char **);
int ProcessCommand(void);
int ShowHelp(char *);
int ShowCfg(char *);
int SetLed(char *);
int DumpOutImage(char *);
int WriteInImage(char *);
int LoopBackIO(char *);
int ReadMsg(char *);
int ShowProcStat(char *);

/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

/*** Backplane ***/
MVIHANDLE  handle;           // Handle returned from MVIbp_Open function
MVIBPIOCONFIG ioconfig;

/*** Serial Port ***/
int Commport;

/***  Commands  ***/
typedef struct
{
    char    *Cmd;
    int     (*CmdFunct)(char *);
} COMMANDSTRUCT;

COMMANDSTRUCT Commands[] =
{
    { "cfg", ShowCfg },
    { "do", DumpOutImage },
    { "wi", WriteInImage },
    { "ps", ShowProcStat },
    { "loop", LoopBackIO },
    { "led", SetLed },
    { "help", ShowHelp },
    { "?", ShowHelp },
    { "quit", NULL },
    { "q", NULL },
    { "rmsg", ReadMsg },
    { "", NULL }
};

WORD ioBuf[256];    // temporary buffer for I/O image data
char pBuf[128];     // temp buffer for messages


/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/***********************************************************************/
void ComPutStr(char *str)
{
    if (MVIsp_Puts(Commport,(BYTE *)str,0,NULL,1000L) != MVI_SUCCESS)
        SPErrorExit();
}

/***********************************************************************/
void ComGetStr(char *str)
{
    BYTE ch;

    do
    {
        if (MVIsp_Getch(Commport,&ch,TIMEOUT_FOREVER) != MVI_SUCCESS)
            SPErrorExit();
        MVIsp_Putch(Commport,ch,1000L);
        if (ch == '\r')
            *str++ = '\0';
        else
            *str++ = ch;
    }
    while(ch != '\r' );
}

/***********************************************************************/
void BPErrorExit(int errcode)
{
    char errbuf[80];

    MVIbp_ErrorString(errcode, errbuf);        // get error message
    ComPutStr(errbuf);
    ComPutStr("\n\r");
    MVIbp_Close(handle);                    // should always close before exiting
    MVIsp_Close(Commport);                    // should always close before exiting
    exit(1);
}

/***********************************************************************/
void SPErrorExit(void)
{
    /* if serial port error, can't rely on error messages */
    MVIbp_Close(handle);                    // should always close before exiting
    MVIsp_Close(Commport);                    // should always close before exiting
    exit(1);
}


/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       main                                           
*
*     Description:                                                      
*       This is the entry point of the sample program.  A variety of
*       API functions are called to illustrate usage.
*
*     Arguments:                                                        
*       none
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       none
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void main(int argc, char *argv[])
{
    int     CmdSts;
    int     rc;
    char    tmpbuf[50];

    Commport = COM1;

    ParseCmdLine(argc,argv);

    rc = MVIsp_Open(Commport,BAUD_19200,PARITY_NONE,WORDLEN8,STOPBITS1);
    if (rc != MVI_SUCCESS)
    {
        printf("Error Opening Serial Port %d\n",rc);
        exit(1);
    }

    ComPutStr("\n\rMVI Sample Program V0.2");
    ComPutStr("\n\r--Enter 'help' or '?' to show menu\n\r");

     // Open the backplane API & get iocfg
    if (MVI_SUCCESS != (rc = MVIbp_Open(&handle)))
    {
        sprintf(tmpbuf,"\n\rMVIbp_Open failed: %d\n\r", rc);
        ComPutStr(tmpbuf);
        BPErrorExit(rc);
    }
    if (MVI_SUCCESS != (rc = MVIbp_GetIOConfig(handle, &ioconfig)))
    {
        sprintf(tmpbuf,"\nMVIbp_GetIOConfig failed: %d\n", rc);
        ComPutStr(tmpbuf);
        BPErrorExit(rc);
    }

    do
    {
        /* display prompt */
        ComPutStr("\n\rmvi> ");

        /* get a command from user */
        CmdSts = ProcessCommand();

    } while (CmdSts == 0);

    // Close the API
    MVIsp_Close(Commport);                    // should always close before exiting
    MVIbp_Close(handle);                    // should always close before exiting
}

/******************************************************************************
******************************************************************************/
int ProcessCommand(void)
{
    static char cCmdBuf[100];
    char    *szArgs;
    char    *szCmd;
    COMMANDSTRUCT *pCommands;
    int CmdSts;

    memset(cCmdBuf,0,100);
    ComGetStr(cCmdBuf);

    /* process command */
    szCmd = strtok(cCmdBuf," \t\r");
    if (strlen(szCmd) == 0)
        return(0);   /* re-prompt if user keeps pressing cr */

    pCommands = (COMMANDSTRUCT *)&Commands;
    while (strlen(pCommands->Cmd) > 0)
    {
        if (stricmp(pCommands->Cmd,szCmd) == 0)
        {
            if (pCommands->CmdFunct != NULL)
            {
                szArgs = &(cCmdBuf[strlen(szCmd)+1]);
                CmdSts = (pCommands->CmdFunct)(szArgs);
                if (CmdSts == 1)
                    ComPutStr("\n\rError #1 -- Bad Syntax");
                else if (CmdSts > 1)
                {
                    sprintf(pBuf,"\n\rError %d",CmdSts);
                    ComPutStr(pBuf);
                }
            }
            else if (szCmd[0] == 'q' || szCmd[0] == 'Q')
                return(1);  /* exit the program */
            return(0);
        }
        pCommands++;
    }
    /* command not found */
    ComPutStr("\n\rHuh? ('?' gives Command List)");
    return(0);
}

/******************************************************************************
******************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
int ShowHelp(char *tmp)
{
    ComPutStr("\n\r\n\rCommands:");
    ComPutStr("\n\r----------------------");
    ComPutStr("\n\rdo [word# len]    - dump output image");
    ComPutStr("\n\rwi word# val      - write val(hex) to input word");
    ComPutStr("\n\rled [1|2] [0|1]   - turn LED 1 or 2 off(0) or on(1)");
    ComPutStr("\n\rcfg               - show the io configuration");
    ComPutStr("\n\rloop              - loopback output word0 to input word0");
    ComPutStr("\n\rrmsg              - display received message");
    ComPutStr("\n\rps                - display processor status");
    ComPutStr("\n\rhelp,?            - print this Command List\n\r");
    return(0);
}
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/******************************************************************************
******************************************************************************/
int DumpOutImage(char *szTmp)
{
    int rc;
    WORD i;
    int wordnum;
    WORD len;
    char *s;

    /* parse command for word# and length */
    s = strtok(szTmp," \t\r");
    if (strlen(s) == 0)
    {
        wordnum = 0;
        len = ioconfig.DirectOutputSize;
    }
    else
    {
        wordnum = atoi(s);
        s = strtok(NULL," \t\r");
        if (strlen(s) == 0)
        {
            len = ioconfig.DirectOutputSize - wordnum;
        }
        len = (WORD)strtoul(s,NULL,10);
    }

    if (MVI_SUCCESS != (rc = MVIbp_ReadOutputImage(handle, ioBuf, wordnum, len)))
    {
        sprintf(pBuf,"\n\rMVIbp_ReadOutputImage failed: %d",rc);
        ComPutStr(pBuf);
        return(0);
    }

    for (i=0; i<len; i++)
    {
        sprintf(pBuf,"\n\rWord %d: %04X",wordnum+i,ioBuf[i]);
        ComPutStr(pBuf);
    }

    return(0);
}

/******************************************************************************
******************************************************************************/
int WriteInImage(char *szTmp)
{
    int rc;
    char    *s;
    int wordnum;
    WORD val;
    char ss[50];

    /* parse command for word# and val */
    s = strtok(szTmp," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    wordnum = atoi(s);
    s = strtok(NULL," \t\r");
    if (strlen(s) == 0)
        return(1);          /* syntax error */
    val = (WORD)strtoul(s,NULL,16);

    if (MVI_SUCCESS != (rc = MVIbp_WriteInputImage(handle, &val, wordnum, 1)))
    {
        sprintf(ss,"\n\rMVIbp_WriteInputImage failed: %d",rc);
        ComPutStr(ss);
    }

    return(0);
}

/******************************************************************************
******************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
int LoopBackIO(char *szTmp)
{
    int rc;
    WORD buf;
    BYTE bch;
    char s[50];

    ComPutStr("\n\rPress a key to end");

    do
    {
        if (MVI_SUCCESS != (rc = MVIbp_ReadOutputImage(handle, &buf, 0, 1)))
        {
            sprintf(s,"\nMVIbp_ReadOutputImage failed: %d\n", rc);
            ComPutStr(s);
            return(0);
        }

        if (MVI_SUCCESS != (rc = MVIbp_WriteInputImage(handle, &buf, 0, 1)))
        {
            sprintf(s,"\nMVIbp_WriteInputImage failed: %d\n", rc);
            ComPutStr(s);
            return(0);
        }
        rc = MVIsp_Getch(Commport,&bch,10L);
    } while (rc == MVI_ERR_TIMEOUT);

    return(0);
}
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/******************************************************************************
******************************************************************************/
//#pragma argsused
int SetLed(char *szTmp)
{
    int led,state;
    char *s;
    int rc;
    char ss[50];

    led = 0;
    state = 0;

    s = strtok(szTmp," \t\r");
    led = atoi(s);
    s = strtok(NULL," \t\r");
    state = atoi(s);

    if ( (led != 1) && (led != 2))
        return(1);
    if ( (state != 0) && (state != 1))
        return(1);

    if (MVI_SUCCESS != (rc = MVIbp_SetUserLED(handle, led, state)))
    {
        sprintf(ss,"\n\rSetUserLed Failed: %d",rc);
        ComPutStr(ss);
    }
    return(0);
}

/********************************************************************
********************************************************************/

#ifdef __BORLANDC__
#pragma warn -par*
#endif
int ShowCfg(char *szTmp)
{
    int rc;
    MVIBPVERSIONINFO verinfo;
    MVISPVERSIONINFO spver;

    // Update the config info (may change when 5550 makes connection)
    if (MVI_SUCCESS != (rc = MVIbp_GetIOConfig(handle, &ioconfig)))
    {
        sprintf(pBuf,"\nMVIbp_GetIOConfig failed: %d\n", rc);
        ComPutStr(pBuf);
        BPErrorExit(rc);
    }

    // Display the Serial Port API series and revision
    if (MVI_SUCCESS != (rc = MVIsp_GetVersionInfo(&spver)))
    {
        sprintf(pBuf,"\nMVIsp_GetVersionInfo failed: %d\n", rc);
        ComPutStr(pBuf);
        return(0);
    }
    sprintf(pBuf,"\n\rSP API Series: %d  Revision: %d", spver.APISeries, spver.APIRevision);
    ComPutStr(pBuf);

    // Display the BP API series and revision
    if (MVI_SUCCESS != (rc = MVIbp_GetVersionInfo(handle, &verinfo)))
    {
        sprintf(pBuf,"\nMVIbp_GetVersionInfo failed: %d\n", rc);
        ComPutStr(pBuf);
        return(0);
    }
    sprintf(pBuf,"\n\rBP API Series: %d  Revision: %d", verinfo.APISeries, verinfo.APIRevision);
    ComPutStr(pBuf);

    // Display the driver series and revision
    sprintf(pBuf,"\n\rBackplane Driver Series: %d  Revision: %d",
                verinfo.BPDDSeries, verinfo.BPDDRevision);
    ComPutStr(pBuf);

    // Display the IO configuration
    sprintf(pBuf,"\n\rInput Words: %d  Output Words: %d\n",
        ioconfig.TotalInputSize, ioconfig.TotalOutputSize);
    ComPutStr(pBuf);
    return(0);
}
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/********************************************************************
********************************************************************/
void ParseCmdLine(int argc, char *argv[])
{
    int     n;
    char    *arg;
    char    ch;
    
    for (n=1; n<argc; n++)
    {
        arg = argv[n];

        if ((*arg == '/') || (*arg == '-'))
        {
            ch = *(++arg);
            ch = toupper(ch);
            arg++;
            switch (ch)
            {
                case 'C':
                    Commport = atoi(arg);
                    break;
            }
        }
    }
}


/******************************************************************************
******************************************************************************/
void DumpMem(BYTE *buf, WORD length)
{
	int		n,i;
	char	ascdmp[17];
    WORD    adr=0;
    char ss[50];
    BYTE    x;

	ascdmp[16] = 0;
	do
	{
		sprintf(ss, "\n\r%03X   ", adr);
        ComPutStr(ss);
		for (i = 0; i < 16; i++)
		{
			x = *(buf+adr);
            adr++;
			sprintf(ss, "%02X ",x);
            ComPutStr(ss);
			ascdmp[i] = ((x > 31) && (x < 127)) ? (char) x : '.';
            if (adr == length)
            {
                ascdmp[i+1] = 0;
                break;
            }
		}
        if (i<16)
        {
            for (n=0; n<(15-i); n++)
            {
			    sprintf(ss, "   ");
                ComPutStr(ss);
            }
        }
		sprintf(ss,"   |%s|",ascdmp);
        ComPutStr(ss);
	}
	while (adr < length);
}


/******************************************************************************
******************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
int ReadMsg(char *szTmp)
{
    int rc;
    char ss[50];
    WORD len=256;

    if (MVI_SUCCESS != (rc = MVIbp_ReceiveMessage(handle, ioBuf, &len, 0, 0)))
    {
        if (rc != MVI_ERR_TIMEOUT)
        {
            sprintf(ss,"\n\rReadMessage Failed: %d",rc);
            ComPutStr(ss);
        }
        else
        {
            sprintf(ss,"\n\rNo message");
            ComPutStr(ss);
        }
    }
    else
    {
        sprintf(ss,"\n\rMessage data:\n\r");
        ComPutStr(ss);
        DumpMem((BYTE *)ioBuf, len*2);
    }
    return(0);
}
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/******************************************************************************
******************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
int ShowProcStat(char *szTmp)
{
    int rc;
    char ss[50];
    WORD ps;

    if (MVI_SUCCESS != (rc = MVIbp_GetProcessorStatus(handle, &ps)))
    {
        sprintf(ss,"\n\rGetProcessorStatus Failed: %d",rc);
        ComPutStr(ss);
    }
    else
    {
        sprintf(ss,"\n\rProcessor Status: %04X\n\r", ps);
        ComPutStr(ss);
    }
    return(0);
}
#ifdef __BORLANDC__
#pragma warn .par*
#endif

